// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { KeyringNode, needs, KeyringTraceFlag, immutableClass, readOnlyProperty, unwrapDataKey, } from '@aws-crypto/material-management-node';
import { randomBytes, createCipheriv, createDecipheriv } from 'crypto';
import { serializeFactory, concatBuffers } from '@aws-crypto/serialize';
import { _onEncrypt, _onDecrypt, NodeRawAesMaterial, rawAesEncryptedDataKeyFactory, rawAesEncryptedPartsFactory, } from '@aws-crypto/raw-keyring';
const fromUtf8 = (input) => Buffer.from(input, 'utf8');
const toUtf8 = (input) => Buffer.from(input.buffer, input.byteOffset, input.byteLength).toString('utf8');
const { serializeEncryptionContext } = serializeFactory(fromUtf8);
const { rawAesEncryptedDataKey } = rawAesEncryptedDataKeyFactory(toUtf8, fromUtf8);
const { rawAesEncryptedParts } = rawAesEncryptedPartsFactory(fromUtf8);
export class RawAesKeyringNode extends KeyringNode {
    constructor(input) {
        super();
        const { keyName, keyNamespace, unencryptedMasterKey, wrappingSuite } = input;
        /* Precondition: AesKeyringNode needs identifying information for encrypt and decrypt. */
        needs(keyName && keyNamespace, 'Identifying information must be defined.');
        /* Precondition: RawAesKeyringNode requires wrappingSuite to be a valid RawAesWrappingSuite. */
        const wrappingMaterial = new NodeRawAesMaterial(wrappingSuite)
            /* Precondition: unencryptedMasterKey must correspond to the NodeAlgorithmSuite specification.
             * Note: the KeyringTrace and flag are _only_ set because I am reusing an existing implementation.
             * See: raw_aes_material.ts in @aws-crypto/raw-keyring for details
             */
            .setUnencryptedDataKey(unencryptedMasterKey, {
            keyNamespace,
            keyName,
            flags: KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY,
        });
        const _wrapKey = async (material) => {
            /* The AAD section is uInt16BE(length) + AAD
             * see: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#header-aad
             * However, the RAW Keyring wants _only_ the ADD.
             * So, I just slice off the length.
             */
            const { buffer, byteOffset, byteLength } = serializeEncryptionContext(material.encryptionContext).slice(2);
            const aad = Buffer.from(buffer, byteOffset, byteLength);
            const { keyNamespace, keyName } = this;
            return aesGcmWrapKey(keyNamespace, keyName, material, aad, wrappingMaterial);
        };
        const _unwrapKey = async (material, edk) => {
            const { keyNamespace, keyName } = this;
            /* The AAD section is uInt16BE(length) + AAD
             * see: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#header-aad
             * However, the RAW Keyring wants _only_ the ADD.
             * So, I just slice off the length.
             */
            const { buffer, byteOffset, byteLength } = serializeEncryptionContext(material.encryptionContext).slice(2);
            const aad = Buffer.from(buffer, byteOffset, byteLength);
            // const aad = Buffer.concat(encodeEncryptionContext(context || {}))
            return aesGcmUnwrapKey(keyNamespace, keyName, material, wrappingMaterial, edk, aad);
        };
        readOnlyProperty(this, 'keyName', keyName);
        readOnlyProperty(this, 'keyNamespace', keyNamespace);
        readOnlyProperty(this, '_wrapKey', _wrapKey);
        readOnlyProperty(this, '_unwrapKey', _unwrapKey);
    }
    _filter({ providerId, providerInfo }) {
        const { keyNamespace, keyName } = this;
        return providerId === keyNamespace && providerInfo.startsWith(keyName);
    }
    _onEncrypt = _onEncrypt(randomBytesAsync);
    _onDecrypt = _onDecrypt();
}
immutableClass(RawAesKeyringNode);
const encryptFlags = KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY |
    KeyringTraceFlag.WRAPPING_KEY_SIGNED_ENC_CTX;
const decryptFlags = KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY |
    KeyringTraceFlag.WRAPPING_KEY_VERIFIED_ENC_CTX;
/**
 * Uses aes-gcm to encrypt the data key and return the passed NodeEncryptionMaterial with
 * an EncryptedDataKey added.
 * @param keyNamespace [String] The keyring namespace (for KeyringTrace)
 * @param keyName [String] The keyring name (for KeyringTrace and to extract the extra info stored in providerInfo)
 * @param material [NodeEncryptionMaterial] The target material to which the EncryptedDataKey will be added
 * @param aad [Buffer] The serialized aad (EncryptionContext)
 * @param wrappingMaterial [NodeRawAesMaterial] The material used to decrypt the EncryptedDataKey
 * @returns [NodeEncryptionMaterial] Mutates and returns the same NodeEncryptionMaterial that was passed but with an EncryptedDataKey added
 */
function aesGcmWrapKey(keyNamespace, keyName, material, aad, wrappingMaterial) {
    const { encryption, ivLength } = wrappingMaterial.suite;
    const iv = randomBytes(ivLength);
    const wrappingDataKey = wrappingMaterial.getUnencryptedDataKey();
    const dataKey = unwrapDataKey(material.getUnencryptedDataKey());
    const cipher = createCipheriv(encryption, wrappingDataKey, iv).setAAD(aad);
    // Buffer.concat will use the shared buffer space, and the resultant buffer will have a byteOffset...
    const ciphertext = concatBuffers(cipher.update(dataKey), cipher.final());
    const authTag = cipher.getAuthTag();
    const edk = rawAesEncryptedDataKey(keyNamespace, keyName, iv, ciphertext, authTag);
    return material.addEncryptedDataKey(edk, encryptFlags);
}
/**
 * Uses aes-gcm to decrypt the encrypted data key and return the passed NodeDecryptionMaterial with
 * the unencrypted data key set.
 * @param keyNamespace [String] The keyring namespace (for KeyringTrace)
 * @param keyName [String] The keyring name (for KeyringTrace and to extract the extra info stored in providerInfo)
 * @param material [NodeDecryptionMaterial] The target material to which the decrypted data key will be added
 * @param wrappingMaterial [NodeRawAesMaterial] The material used to decrypt the EncryptedDataKey
 * @param edk [EncryptedDataKey] The EncryptedDataKey on which to operate
 * @param aad [Buffer] The serialized aad (EncryptionContext)
 * @returns [NodeDecryptionMaterial] Mutates and returns the same NodeDecryptionMaterial that was passed but with the unencrypted data key set
 */
function aesGcmUnwrapKey(keyNamespace, keyName, material, wrappingMaterial, edk, aad) {
    const { authTag, ciphertext, iv } = rawAesEncryptedParts(material.suite, keyName, edk);
    const { encryption } = wrappingMaterial.suite;
    // createDecipheriv is incorrectly typed in @types/node. It should take key: CipherKey, not key: BinaryLike
    const decipher = createDecipheriv(encryption, wrappingMaterial.getUnencryptedDataKey(), iv)
        .setAAD(aad)
        .setAuthTag(authTag);
    // Buffer.concat will use the shared buffer space, and the resultant buffer will have a byteOffset...
    const unencryptedDataKey = concatBuffers(decipher.update(ciphertext), decipher.final());
    const trace = { keyNamespace, keyName, flags: decryptFlags };
    return material.setUnencryptedDataKey(unencryptedDataKey, trace);
}
async function randomBytesAsync(size) {
    return new Promise((resolve, reject) => {
        randomBytes(size, (err, buffer) => {
            if (err)
                return reject(err);
            resolve(buffer);
        });
    });
}
//# sourceMappingURL=data:application/json;base64,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